const express = require("express");
const router = express.Router();
const mongoose = require("mongoose");
const asyncHandler = require("express-async-handler");
const Genre = require("../model/Genre");

// CREATE Genre
router.post(
  "/create",
  asyncHandler(async (req, res) => {
    const { name } = req.body;

    if (!name || typeof name !== "string" || !name.trim()) {
      res.status(400);
      throw new Error("Genre name is required");
    }

    const existing = await Genre.findOne({ name: name.trim() });
    if (existing) {
      res.status(409);
      throw new Error("Genre already exists");
    }

    const genre = await Genre.create({ name: name.trim() });

    res.status(201).json({
      success: true,
      message: "Genre created successfully",
      genre,
    });
  })
);

// READ All Genres
router.get(
  "/fetch",
  asyncHandler(async (req, res) => {
    const genres = await Genre.find().sort({ createdAt: -1 });
    res.status(200).json({
      success: true,
      count: genres.length,
      genres,
    });
  })
);

// READ One Genre
router.get(
  "fetch/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Genre ID");
    }

    const genre = await Genre.findById(id);
    if (!genre) {
      res.status(404);
      throw new Error("Genre not found");
    }

    res.status(200).json({ success: true, genre });
  })
);

// UPDATE Genre
router.put(
  "edit/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;
    const { name } = req.body;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Genre ID");
    }

    if (!name || typeof name !== "string" || !name.trim()) {
      res.status(400);
      throw new Error("Genre name is required");
    }

    const genre = await Genre.findByIdAndUpdate(
      id,
      { name: name.trim() },
      { new: true, runValidators: true }
    );

    if (!genre) {
      res.status(404);
      throw new Error("Genre not found");
    }

    res.status(200).json({
      success: true,
      message: "Genre updated successfully",
      genre,
    });
  })
);

// DELETE Genre
router.delete(
  "delete/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Genre ID");
    }

    const genre = await Genre.findByIdAndDelete(id);
    if (!genre) {
      res.status(404);
      throw new Error("Genre not found");
    }

    res.status(200).json({
      success: true,
      message: "Genre deleted successfully",
    });
  })
);

module.exports = router;
