const express = require("express");
const router = express.Router();
const mongoose = require("mongoose");
const asyncHandler = require("express-async-handler");
const Comment = require("../model/Comment");
const Blog = require("../model/Blog");

// CREATE Comment
router.post(
  "/create",
  asyncHandler(async (req, res) => {
    const { content, blogId, parentCommentId } = req.body;

    // Validation
    if (!content || typeof content !== "string" || !content.trim()) {
      res.status(400);
      throw new Error("Comment content is required");
    }

    if (!blogId || !mongoose.Types.ObjectId.isValid(blogId)) {
      res.status(400);
      throw new Error("Valid Blog ID is required");
    }

    // Check if blog exists
    const blog = await Blog.findById(blogId);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // If it's a reply, check if parent comment exists
    if (parentCommentId) {
      if (!mongoose.Types.ObjectId.isValid(parentCommentId)) {
        res.status(400);
        throw new Error("Invalid parent comment ID");
      }

      const parentComment = await Comment.findById(parentCommentId);
      if (!parentComment) {
        res.status(404);
        throw new Error("Parent comment not found");
      }

      // Ensure parent comment belongs to the same blog
      if (parentComment.blog.toString() !== blogId) {
        res.status(400);
        throw new Error("Parent comment does not belong to this blog");
      }
    }

    // Create comment
    const comment = await Comment.create({
      content: content.trim(),
      author: req.user._id,
      blog: blogId,
      parentComment: parentCommentId || null,
    });

    const populatedComment = await Comment.findById(comment._id)
      .populate("author", "username firstName lastName pfp")
      .populate("parentComment", "content author");

    res.status(201).json({
      success: true,
      message: parentCommentId
        ? "Reply added successfully"
        : "Comment added successfully",
      comment: populatedComment,
    });
  })
);

// READ All Comments for a Blog
router.get(
  "/fetch/blog/:blogId",
  asyncHandler(async (req, res) => {
    const { blogId } = req.params;
    const { page = 1, limit = 20 } = req.query;

    if (!mongoose.Types.ObjectId.isValid(blogId)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    // Check if blog exists
    const blog = await Blog.findById(blogId);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);

    // Fetch only top-level comments (parentComment is null)
    const [comments, total] = await Promise.all([
      Comment.find({ blog: blogId, parentComment: null })
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(parseInt(limit))
        .populate("author", "username firstName lastName pfp")
        .lean(),
      Comment.countDocuments({ blog: blogId, parentComment: null }),
    ]);

    // Fetch replies for each top-level comment
    const commentsWithReplies = await Promise.all(
      comments.map(async (comment) => {
        const replies = await Comment.find({ parentComment: comment._id })
          .sort({ createdAt: 1 })
          .populate("author", "username firstName lastName pfp")
          .lean();

        return {
          ...comment,
          replies,
          replyCount: replies.length,
        };
      })
    );

    res.status(200).json({
      success: true,
      count: comments.length,
      total,
      page: parseInt(page),
      totalPages: Math.ceil(total / parseInt(limit)),
      comments: commentsWithReplies,
    });
  })
);

// READ Replies for a Comment
router.get(
  "/fetch/replies/:commentId",
  asyncHandler(async (req, res) => {
    const { commentId } = req.params;

    if (!mongoose.Types.ObjectId.isValid(commentId)) {
      res.status(400);
      throw new Error("Invalid Comment ID");
    }

    const parentComment = await Comment.findById(commentId);
    if (!parentComment) {
      res.status(404);
      throw new Error("Comment not found");
    }

    const replies = await Comment.find({ parentComment: commentId })
      .sort({ createdAt: 1 })
      .populate("author", "username firstName lastName pfp");

    res.status(200).json({
      success: true,
      count: replies.length,
      replies,
    });
  })
);

// READ One Comment
router.get(
  "/fetch/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Comment ID");
    }

    const comment = await Comment.findById(id)
      .populate("author", "username firstName lastName pfp occupation")
      .populate("blog", "title slug")
      .populate("parentComment", "content author")
      .populate("likes", "username");

    if (!comment) {
      res.status(404);
      throw new Error("Comment not found");
    }

    res.status(200).json({ success: true, comment });
  })
);

// UPDATE Comment
router.put(
  "/edit/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;
    const { content } = req.body;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Comment ID");
    }

    if (!content || typeof content !== "string" || !content.trim()) {
      res.status(400);
      throw new Error("Comment content is required");
    }

    const comment = await Comment.findById(id);
    if (!comment) {
      res.status(404);
      throw new Error("Comment not found");
    }

    // Check if user is the comment author
    if (comment.author.toString() !== req.user._id.toString()) {
      res.status(403);
      throw new Error("You can only edit your own comments");
    }

    comment.content = content.trim();
    comment.isEdited = true;
    comment.editedAt = new Date();
    await comment.save();

    const updatedComment = await Comment.findById(id)
      .populate("author", "username firstName lastName pfp")
      .populate("parentComment", "content author");

    res.status(200).json({
      success: true,
      message: "Comment updated successfully",
      comment: updatedComment,
    });
  })
);

// DELETE Comment
router.delete(
  "/delete/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Comment ID");
    }

    const comment = await Comment.findById(id);
    if (!comment) {
      res.status(404);
      throw new Error("Comment not found");
    }

    // Check if user is the comment author or admin
    if (
      comment.author.toString() !== req.user._id.toString() &&
      req.user.role !== "admin"
    ) {
      res.status(403);
      throw new Error("You can only delete your own comments");
    }

    // Delete all replies to this comment
    await Comment.deleteMany({ parentComment: id });

    // Delete the comment itself
    await Comment.findByIdAndDelete(id);

    res.status(200).json({
      success: true,
      message: "Comment deleted successfully",
    });
  })
);

// LIKE/UNLIKE Comment
router.post(
  "/like/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Comment ID");
    }

    const comment = await Comment.findById(id);
    if (!comment) {
      res.status(404);
      throw new Error("Comment not found");
    }

    const userId = req.user._id;
    const hasLiked = comment.likes.includes(userId);

    if (hasLiked) {
      // Unlike
      comment.likes = comment.likes.filter(
        (like) => like.toString() !== userId.toString()
      );
    } else {
      // Like
      comment.likes.push(userId);
    }

    await comment.save();

    res.status(200).json({
      success: true,
      message: hasLiked ? "Comment unliked" : "Comment liked",
      likes: comment.likes.length,
      hasLiked: !hasLiked,
    });
  })
);

// GET User's Comments
router.get(
  "/fetch/user/:userId",
  asyncHandler(async (req, res) => {
    const { userId } = req.params;
    const { page = 1, limit = 10 } = req.query;

    if (!mongoose.Types.ObjectId.isValid(userId)) {
      res.status(400);
      throw new Error("Invalid User ID");
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);

    const [comments, total] = await Promise.all([
      Comment.find({ author: userId })
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(parseInt(limit))
        .populate("blog", "title slug coverImage")
        .populate("parentComment", "content"),
      Comment.countDocuments({ author: userId }),
    ]);

    res.status(200).json({
      success: true,
      count: comments.length,
      total,
      page: parseInt(page),
      totalPages: Math.ceil(total / parseInt(limit)),
      comments,
    });
  })
);

module.exports = router;
