const express = require("express");
const router = express.Router();
const asyncHandler = require("express-async-handler");
const mongoose = require("mongoose");
const Blog = require("../model/Blog");
const { uploadImage } = require("../helper/upload");
const { isBlogger } = require("../middleware/bloggerAuth");
const fs = require("fs");
const path = require("path");

// Helper function to generate slug from title
const generateSlug = (title) => {
  return title
    .toLowerCase()
    .trim()
    .replace(/[^\w\s-]/g, "")
    .replace(/[\s_-]+/g, "-")
    .replace(/^-+|-+$/g, "");
};

// CREATE Blog (with image upload) - Only Blogger/Admin
router.post(
  "/create",
  isBlogger,
  uploadImage.single("coverImage"),
  asyncHandler(async (req, res) => {
    const { title, description, content, genres, tags } = req.body;

    // Validation
    if (!title || typeof title !== "string" || !title.trim()) {
      res.status(400);
      throw new Error("Blog title is required");
    }

    if (
      !description ||
      typeof description !== "string" ||
      !description.trim()
    ) {
      res.status(400);
      throw new Error("Blog description is required");
    }

    if (!content || typeof content !== "string" || !content.trim()) {
      res.status(400);
      throw new Error("Blog content is required");
    }

    // Generate unique slug
    let slug = generateSlug(title);
    const existingSlug = await Blog.findOne({ slug });
    if (existingSlug) {
      slug = `${slug}-${Date.now()}`;
    }

    // Parse genres and tags if they're strings
    let parsedGenres = [];
    let parsedTags = [];

    if (genres) {
      parsedGenres = typeof genres === "string" ? JSON.parse(genres) : genres;
    }

    if (tags) {
      parsedTags = typeof tags === "string" ? JSON.parse(tags) : tags;
    }

    // Get image path if uploaded
    const coverImage = req.file ? `/uploads/images/${req.file.filename}` : null;

    // Create blog
    const blog = await Blog.create({
      title: title.trim(),
      slug,
      description: description.trim(),
      content: content.trim(),
      coverImage,
      author: req.user.id,
      genres: parsedGenres,
      tags: parsedTags,
    });

    const populatedBlog = await Blog.findById(blog.id)
      .populate("author", "username email firstName lastName role")
      .populate("genres", "name");

    res.status(201).json({
      success: true,
      message: "Blog created successfully",
      blog: populatedBlog,
    });
  })
);

// READ All Blogs (with filters and pagination) - Public
router.get(
  "/fetch",
  asyncHandler(async (req, res) => {
    const { page = 1, limit = 10, author, genre, search } = req.query;

    // Build query
    const query = {};

    if (author && mongoose.Types.ObjectId.isValid(author))
      query.author = author;
    if (genre && mongoose.Types.ObjectId.isValid(genre)) query.genres = genre;
    if (search) {
      query.$or = [
        { title: { $regex: search, $options: "i" } },
        { description: { $regex: search, $options: "i" } },
        { tags: { $regex: search, $options: "i" } },
      ];
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);

    const [blogs, total] = await Promise.all([
      Blog.find(query)
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(parseInt(limit))
        .populate("author", "username email firstName lastName pfp role")
        .populate("genres", "name")
        .populate("likes", "username firstName lastName"),
      Blog.countDocuments(query),
    ]);

    res.status(200).json({
      success: true,
      count: blogs.length,
      total,
      page: parseInt(page),
      totalPages: Math.ceil(total / parseInt(limit)),
      blogs,
    });
  })
);

// READ Blogs by User ID - Public
router.get(
  "/fetch/user/:userId",
  asyncHandler(async (req, res) => {
    const { userId } = req.params;
    const { page = 1, limit = 10 } = req.query;

    if (!mongoose.Types.ObjectId.isValid(userId)) {
      res.status(400);
      throw new Error("Invalid User ID");
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);

    const [blogs, total] = await Promise.all([
      Blog.find({ author: userId })
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(parseInt(limit))
        .populate(
          "author",
          "username email firstName lastName pfp occupation role"
        )
        .populate("genres", "name")
        .populate("likes", "username firstName lastName"),
      Blog.countDocuments({ author: userId }),
    ]);

    if (!blogs || blogs.length === 0) {
      res.status(404);
      throw new Error("No blogs found for this user");
    }

    res.status(200).json({
      success: true,
      count: blogs.length,
      total,
      page: parseInt(page),
      totalPages: Math.ceil(total / parseInt(limit)),
      blogs,
    });
  })
);

// READ One Blog by ID - Public
router.get(
  "/fetch/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id)
      .populate("author", "username email firstName lastName pfp occupation")
      .populate("genres", "name")
      .populate("likes", "username firstName lastName");

    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    res.status(200).json({ success: true, blog });
  })
);

// READ One Blog by Slug - Public
router.get(
  "/fetch/slug/:slug",
  asyncHandler(async (req, res) => {
    const { slug } = req.params;

    const blog = await Blog.findOne({ slug })
      .populate("author", "username email firstName lastName pfp occupation")
      .populate("genres", "name")
      .populate("likes", "username firstName lastName");

    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    res.status(200).json({ success: true, blog });
  })
);

// UPDATE Blog - Only Author or Admin
router.put(
  "/edit/:id",
  isBlogger,
  uploadImage.single("coverImage"),
  asyncHandler(async (req, res) => {
    const { id } = req.params;
    const { title, description, content, genres, tags } = req.body;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Update fields
    const updateData = {};

    if (title) {
      updateData.title = title.trim();
      updateData.slug = generateSlug(title);

      // Check if new slug already exists
      const existingSlug = await Blog.findOne({
        slug: updateData.slug,
        _id: { $ne: id },
      });
      if (existingSlug) {
        updateData.slug = `${updateData.slug}-${Date.now()}`;
      }
    }

    if (description) updateData.description = description.trim();
    if (content) updateData.content = content.trim();
    if (genres) {
      updateData.genres =
        typeof genres === "string" ? JSON.parse(genres) : genres;
    }
    if (tags) {
      updateData.tags = typeof tags === "string" ? JSON.parse(tags) : tags;
    }

    // Handle image update
    if (req.file) {
      if (blog.coverImage) {
        const oldImagePath = path.join(
          __dirname,
          "..",
          "views",
          blog.coverImage
        );
        if (fs.existsSync(oldImagePath)) {
          fs.unlinkSync(oldImagePath);
        }
      }
      updateData.coverImage = `/uploads/images/${req.file.filename}`;
    }

    const updatedBlog = await Blog.findByIdAndUpdate(id, updateData, {
      new: true,
      runValidators: true,
    })
      .populate("author", "username email firstName lastName role")
      .populate("genres", "name");

    res.status(200).json({
      success: true,
      message: "Blog updated successfully",
      blog: updatedBlog,
    });
  })
);

// DELETE Blog - Only Author or Admin
router.delete(
  "/delete/:id",
  isBlogger,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    if (blog.coverImage) {
      const imagePath = path.join(__dirname, "..", "views", blog.coverImage);
      if (fs.existsSync(imagePath)) {
        fs.unlinkSync(imagePath);
      }
    }

    await Blog.findByIdAndDelete(id);

    res.status(200).json({
      success: true,
      message: "Blog deleted successfully",
    });
  })
);

// LIKE/UNLIKE Blog
router.post(
  "/like/:id",
  isBlogger,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    const userId = req.user._id;
    const hasLiked = blog.likes.includes(userId);

    if (hasLiked) {
      blog.likes = blog.likes.filter(
        (like) => like.toString() !== userId.toString()
      );
    } else {
      blog.likes.push(userId);
    }

    await blog.save();

    res.status(200).json({
      success: true,
      message: hasLiked ? "Blog unliked" : "Blog liked",
      likes: blog.likes.length,
      hasLiked: !hasLiked,
    });
  })
);

module.exports = router;
